// config/auth.js - Configuração de Autenticação
const jwt = require('jsonwebtoken');
const bcrypt = require('bcryptjs');

// Configurações JWT
const JWT_SECRET = process.env.JWT_SECRET || 'escola_chat_jwt_secret_key';
const JWT_EXPIRES_IN = process.env.JWT_EXPIRES_IN || '7d';

// Configurações de hash de senha
const SALT_ROUNDS = 12;

// Função para gerar hash da senha
async function hashPassword(password) {
    try {
        const salt = await bcrypt.genSalt(SALT_ROUNDS);
        const hash = await bcrypt.hash(password, salt);
        return hash;
    } catch (error) {
        console.error('Erro ao gerar hash da senha:', error);
        throw new Error('Erro ao processar senha');
    }
}

// Função para verificar senha
async function verifyPassword(password, hash) {
    try {
        const isValid = await bcrypt.compare(password, hash);
        return isValid;
    } catch (error) {
        console.error('Erro ao verificar senha:', error);
        return false;
    }
}

// Função para gerar token JWT
function generateToken(payload) {
    try {
        const token = jwt.sign(payload, JWT_SECRET, {
            expiresIn: JWT_EXPIRES_IN,
            issuer: 'escola_chat',
            audience: 'escola_chat_users'
        });
        return token;
    } catch (error) {
        console.error('Erro ao gerar token:', error);
        throw new Error('Erro ao gerar token de autenticação');
    }
}

// Função para verificar token JWT
function verifyToken(token) {
    try {
        const decoded = jwt.verify(token, JWT_SECRET, {
            issuer: 'escola_chat',
            audience: 'escola_chat_users'
        });
        return decoded;
    } catch (error) {
        if (error.name === 'TokenExpiredError') {
            throw new Error('Token expirado');
        } else if (error.name === 'JsonWebTokenError') {
            throw new Error('Token inválido');
        } else {
            console.error('Erro ao verificar token:', error);
            throw new Error('Erro na verificação do token');
        }
    }
}

// Função para decodificar token sem verificar (para debug)
function decodeToken(token) {
    try {
        const decoded = jwt.decode(token);
        return decoded;
    } catch (error) {
        console.error('Erro ao decodificar token:', error);
        return null;
    }
}

// Função para validar força da senha
function validatePasswordStrength(password) {
    const errors = [];
    
    if (!password) {
        errors.push('Senha é obrigatória');
        return { isValid: false, errors };
    }
    
    if (password.length < 6) {
        errors.push('Senha deve ter pelo menos 6 caracteres');
    }
    
    if (password.length > 128) {
        errors.push('Senha muito longa (máximo 128 caracteres)');
    }
    
    // Verificar se tem pelo menos uma letra
    if (!/[a-zA-Z]/.test(password)) {
        errors.push('Senha deve conter pelo menos uma letra');
    }
    
    // Verificar se tem pelo menos um número
    if (!/\d/.test(password)) {
        errors.push('Senha deve conter pelo menos um número');
    }
    
    return {
        isValid: errors.length === 0,
        errors
    };
}

// Função para validar email
function validateEmail(email) {
    const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
    return emailRegex.test(email);
}

// Função para validar telefone brasileiro
function validatePhone(phone) {
    if (!phone) return false;
    
    // Remover caracteres não numéricos
    const cleanPhone = phone.replace(/\D/g, '');
    
    // Verificar se tem 10 ou 11 dígitos (com DDD)
    if (cleanPhone.length !== 10 && cleanPhone.length !== 11) {
        return false;
    }
    
    // Verificar DDD válido (11-99)
    const ddd = parseInt(cleanPhone.substring(0, 2));
    if (ddd < 11 || ddd > 99) {
        return false;
    }
    
    return true;
}

// Função para formatar telefone
function formatPhone(phone) {
    if (!phone) return '';
    
    const cleanPhone = phone.replace(/\D/g, '');
    
    if (cleanPhone.length === 10) {
        return cleanPhone.replace(/(\d{2})(\d{4})(\d{4})/, '($1) $2-$3');
    } else if (cleanPhone.length === 11) {
        return cleanPhone.replace(/(\d{2})(\d{5})(\d{4})/, '($1) $2-$3');
    }
    
    return phone;
}

// Função para limpar telefone (apenas números)
function cleanPhone(phone) {
    if (!phone) return '';
    return phone.replace(/\D/g, '');
}

// Função para validar nome
function validateName(name) {
    if (!name) return false;
    
    // Remover espaços extras
    const cleanName = name.trim();
    
    // Verificar se tem pelo menos 2 caracteres
    if (cleanName.length < 2) return false;
    
    // Verificar se tem pelo menos 2 palavras
    const words = cleanName.split(' ').filter(word => word.length > 0);
    if (words.length < 2) return false;
    
    // Verificar se contém apenas letras, espaços e acentos
    const nameRegex = /^[a-zA-ZÀ-ÿ\s]+$/;
    return nameRegex.test(cleanName);
}

// Função para validar dados de usuário
function validateUserData(userData) {
    const errors = [];
    
    // Validar nome
    if (!validateName(userData.name)) {
        errors.push('Nome deve conter pelo menos 2 palavras e apenas letras');
    }
    
    // Validar email
    if (!validateEmail(userData.email)) {
        errors.push('Email inválido');
    }
    
    // Validar telefone (opcional)
    if (userData.phone && !validatePhone(userData.phone)) {
        errors.push('Telefone inválido');
    }
    
    // Validar WhatsApp (opcional)
    if (userData.whatsapp && !validatePhone(userData.whatsapp)) {
        errors.push('WhatsApp inválido');
    }
    
    // Validar papel do usuário
    const validRoles = ['admin', 'professor', 'student', 'parent', 'director', 'coordinator', 'driver'];
    if (!validRoles.includes(userData.role)) {
        errors.push('Papel de usuário inválido');
    }
    
    return {
        isValid: errors.length === 0,
        errors
    };
}

// Função para gerar código de verificação
function generateVerificationCode() {
    return Math.floor(100000 + Math.random() * 900000).toString();
}

// Função para gerar código de certificado
function generateCertificateCode() {
    const timestamp = Date.now().toString(36);
    const random = Math.random().toString(36).substring(2, 8);
    return `CERT-${timestamp}-${random}`.toUpperCase();
}

// Função para criar slug a partir do texto
function createSlug(text) {
    return text
        .toLowerCase()
        .normalize('NFD')
        .replace(/[\u0300-\u036f]/g, '') // Remove acentos
        .replace(/[^a-z0-9\s-]/g, '') // Remove caracteres especiais
        .replace(/\s+/g, '-') // Substitui espaços por hífens
        .replace(/-+/g, '-') // Remove hífens duplicados
        .replace(/^-|-$/g, ''); // Remove hífens do início e fim
}

// Função para sanitizar dados de entrada
function sanitizeInput(input) {
    if (typeof input !== 'string') return input;
    
    return input
        .trim()
        .replace(/[<>]/g, '') // Remove < e >
        .replace(/javascript:/gi, '') // Remove javascript:
        .replace(/on\w+=/gi, ''); // Remove eventos on*
}

// Função para validar permissões
function hasPermission(userRole, requiredPermission) {
    const permissions = {
        admin: ['all'],
        director: ['manage_school', 'manage_users', 'manage_groups', 'view_reports'],
        coordinator: ['manage_groups', 'moderate_chat', 'view_reports'],
        professor: ['chat', 'view_courses'],
        parent: ['chat'],
        driver: ['chat'],
        student: ['chat', 'view_courses']
    };
    
    const userPermissions = permissions[userRole] || [];
    
    return userPermissions.includes('all') || userPermissions.includes(requiredPermission);
}

// Função para verificar se pode conversar com outro usuário
function canChatWith(userRole, userSchoolId, targetRole, targetSchoolId) {
    // Mesma escola é requisito básico (exceto admins)
    if (userRole !== 'admin' && userSchoolId !== targetSchoolId) {
        return false;
    }
    
    // Regras específicas por papel
    const chatRules = {
        admin: () => true, // Admin pode conversar com todos
        director: (target) => true, // Diretor pode conversar com todos da escola
        coordinator: (target) => true, // Coordenador pode conversar com todos da escola
        professor: (target) => ['parent', 'director', 'coordinator', 'professor'].includes(target),
        parent: (target) => ['professor', 'director', 'coordinator', 'driver'].includes(target),
        driver: (target) => ['parent', 'director', 'coordinator'].includes(target),
        student: (target) => ['professor', 'director', 'coordinator'].includes(target)
    };
    
    const canChat = chatRules[userRole];
    return canChat ? canChat(targetRole) : false;
}

// Função para verificar se pode criar grupos
function canCreateGroups(userRole) {
    return ['admin', 'director', 'coordinator'].includes(userRole);
}

// Função para verificar se pode gerenciar usuários
function canManageUsers(userRole) {
    return ['admin', 'director'].includes(userRole);
}

module.exports = {
    hashPassword,
    verifyPassword,
    generateToken,
    verifyToken,
    decodeToken,
    validatePasswordStrength,
    validateEmail,
    validatePhone,
    formatPhone,
    cleanPhone,
    validateName,
    validateUserData,
    generateVerificationCode,
    generateCertificateCode,
    createSlug,
    sanitizeInput,
    hasPermission,
    canChatWith,
    canCreateGroups,
    canManageUsers
};