// config/whatsapp.js - Configuração do WhatsApp usando wppconnect
const { create, Whatsapp } = require('@wppconnect-team/wppconnect');
const qrcode = require('qrcode-terminal');
const { query, findOne, update, insert } = require('./database');

let client = null;
let isConnected = false;
let currentQR = null;

// Configurações do WhatsApp
const whatsappConfig = {
    session: process.env.WHATSAPP_SESSION_NAME || 'escola_chat',
    headless: false, // Abrir com interface gráfica
    devtools: false,
    useChrome: true,
    debug: false,
    logQR: true,
    browserWS: '',
    browserArgs: [
        '--no-sandbox',
        '--disable-setuid-sandbox',
        '--disable-dev-shm-usage',
        '--disable-accelerated-2d-canvas',
        '--no-first-run',
        '--no-zygote'
        // Removido '--disable-gpu' para melhor compatibilidade
    ],
    puppeteerOptions: {
        headless: false, // Forçar modo com interface
        args: [
            '--no-sandbox',
            '--disable-setuid-sandbox',
            '--disable-web-security',
            '--disable-features=VizDisplayCompositor'
        ]
    },
    disableWelcome: true,
    updatesLog: true,
    autoClose: 60000,
    createPathFileToken: false,
};

// Função para inicializar o WhatsApp
async function initializeWhatsApp() {
    try {
        console.log('📱 Iniciando cliente WhatsApp...');
        
        client = await create({
            session: whatsappConfig.session,
            catchQR: (base64Qr, asciiQR, attempts, urlCode) => {
                console.log('📱 QR Code gerado. Tentativa:', attempts);
                
                // Mostrar QR no terminal
                if (process.env.NODE_ENV !== 'production') {
                    console.log(asciiQR);
                }
                
                // Salvar QR no banco para exibir no admin
                currentQR = base64Qr;
                saveQRToDatabase(base64Qr);
            },
            statusFind: (statusSession, session) => {
                console.log('📱 Status da sessão:', statusSession, session);
                
                if (statusSession === 'qrReadSuccess') {
                    console.log('✅ QR Code lido com sucesso!');
                }
                
                if (statusSession === 'chatsAvailable') {
                    console.log('✅ WhatsApp conectado e pronto!');
                    isConnected = true;
                    updateConnectionStatus(true);
                }
                
                if (statusSession === 'disconnected') {
                    console.log('❌ WhatsApp desconectado');
                    isConnected = false;
                    updateConnectionStatus(false);
                }
            },
            logQR: whatsappConfig.logQR,
            browserArgs: whatsappConfig.browserArgs,
            headless: whatsappConfig.headless,
            devtools: whatsappConfig.devtools,
            useChrome: whatsappConfig.useChrome,
            debug: whatsappConfig.debug,
            browserWS: whatsappConfig.browserWS,
            puppeteerOptions: whatsappConfig.puppeteerOptions,
            disableWelcome: whatsappConfig.disableWelcome,
            updatesLog: whatsappConfig.updatesLog,
            autoClose: whatsappConfig.autoClose,
            createPathFileToken: whatsappConfig.createPathFileToken
        });
        
        // Configurar eventos do cliente
        setupWhatsAppEvents();
        
        return client;
        
    } catch (error) {
        console.error('❌ Erro ao inicializar WhatsApp:', error);
        throw error;
    }
}

// Configurar eventos do WhatsApp
function setupWhatsAppEvents() {
    if (!client) return;
    
    // Evento quando recebe mensagem
    client.onMessage(async (message) => {
        try {
            // Log da mensagem recebida (opcional)
            console.log('📱 Mensagem recebida:', {
                from: message.from,
                body: message.body,
                timestamp: new Date()
            });
            
            // Resposta automática informando que não devem responder
            if (message.body && !message.fromMe) {
                const autoReplyMessage = await getAutoReplyMessage();
                if (autoReplyMessage) {
                    // Aguardar um pouco antes de responder
                    setTimeout(async () => {
                        try {
                            await client.sendText(message.from, autoReplyMessage);
                        } catch (error) {
                            console.error('Erro ao enviar resposta automática:', error);
                        }
                    }, 2000);
                }
            }
            
        } catch (error) {
            console.error('Erro ao processar mensagem recebida:', error);
        }
    });
    
    // Evento de desconexão
    client.onStateChange((state) => {
        console.log('📱 Estado mudou:', state);
        isConnected = state === 'CONNECTED';
        updateConnectionStatus(isConnected);
    });
}

// Função para enviar notificação WhatsApp
async function sendWhatsAppNotification(phoneNumber, message = null) {
    if (!client || !isConnected) {
        throw new Error('WhatsApp não está conectado');
    }
    
    try {
        // Limpar e formatar número
        const cleanPhone = phoneNumber.replace(/\D/g, '');
        const formattedPhone = cleanPhone.includes('@c.us') ? cleanPhone : `55${cleanPhone}@c.us`;
        
        // Usar mensagem personalizada ou padrão
        const notificationMessage = message || await getNotificationMessage();
        
        // Enviar mensagem
        const result = await client.sendText(formattedPhone, notificationMessage);
        
        console.log('✅ Notificação WhatsApp enviada:', {
            phone: formattedPhone,
            message: notificationMessage,
            timestamp: new Date()
        });
        
        return {
            success: true,
            messageId: result.id,
            phone: formattedPhone
        };
        
    } catch (error) {
        console.error('❌ Erro ao enviar notificação WhatsApp:', error);
        throw error;
    }
}

// Função para verificar se um número existe no WhatsApp
async function checkWhatsAppNumber(phoneNumber) {
    if (!client || !isConnected) {
        return false;
    }
    
    try {
        const cleanPhone = phoneNumber.replace(/\D/g, '');
        const formattedPhone = `55${cleanPhone}@c.us`;
        
        const result = await client.checkNumberStatus(formattedPhone);
        return result.canReceiveMessage;
        
    } catch (error) {
        console.error('Erro ao verificar número WhatsApp:', error);
        return false;
    }
}

// Função para obter mensagem de notificação do banco
async function getNotificationMessage() {
    try {
        const config = await findOne('whatsapp_config', { active: 1 });
        return config?.notification_message || 'Você pode ter mensagens novas na plataforma, não responda essa mensagem';
    } catch (error) {
        console.error('Erro ao buscar mensagem de notificação:', error);
        return 'Você pode ter mensagens novas na plataforma, não responda essa mensagem';
    }
}

// Função para obter mensagem de resposta automática
async function getAutoReplyMessage() {
    try {
        const config = await findOne('whatsapp_config', { active: 1 });
        return config?.auto_reply_message || null;
    } catch (error) {
        console.error('Erro ao buscar mensagem de resposta automática:', error);
        return null;
    }
}

// Função para salvar QR no banco
async function saveQRToDatabase(qrCode) {
    try {
        await query(`
            UPDATE whatsapp_config 
            SET qr_code = ?, updated_at = NOW() 
            WHERE session_name = ?
        `, [qrCode, whatsappConfig.session]);
    } catch (error) {
        console.error('Erro ao salvar QR no banco:', error);
    }
}

// Função para atualizar status de conexão
async function updateConnectionStatus(connected) {
    try {
        const now = new Date();
        await query(`
            UPDATE whatsapp_config 
            SET 
                is_connected = ?, 
                last_connection = ?, 
                qr_code = CASE WHEN ? THEN NULL ELSE qr_code END,
                updated_at = NOW()
            WHERE session_name = ?
        `, [connected, now, connected, whatsappConfig.session]);
        
        console.log(`📱 Status de conexão atualizado: ${connected ? 'CONECTADO' : 'DESCONECTADO'}`);
    } catch (error) {
        console.error('Erro ao atualizar status de conexão:', error);
    }
}

// Função para processar fila de notificações
async function processNotificationQueue() {
    if (!client || !isConnected) {
        return;
    }
    
    try {
        // Buscar notificações pendentes
        const pendingNotifications = await query(`
            SELECT * FROM whatsapp_notifications 
            WHERE sent = 0 AND attempts < 3
            ORDER BY created_at ASC 
            LIMIT 10
        `);
        
        for (const notification of pendingNotifications) {
            try {
                // Tentar enviar notificação
                await sendWhatsAppNotification(notification.phone_number);
                
                // Marcar como enviada
                await update('whatsapp_notifications', {
                    sent: 1,
                    sent_at: new Date(),
                    error_message: null
                }, { id: notification.id });
                
                console.log(`✅ Notificação enviada: ID ${notification.id}`);
                
                // Aguardar entre envios para evitar spam
                await new Promise(resolve => setTimeout(resolve, 2000));
                
            } catch (error) {
                // Incrementar tentativas e salvar erro
                await update('whatsapp_notifications', {
                    attempts: notification.attempts + 1,
                    error_message: error.message
                }, { id: notification.id });
                
                console.error(`❌ Erro ao enviar notificação ID ${notification.id}:`, error.message);
            }
        }
        
    } catch (error) {
        console.error('Erro ao processar fila de notificações:', error);
    }
}

// Função para adicionar notificação à fila
async function queueNotification(userId, messageId, phoneNumber) {
    try {
        // Verificar se já existe notificação pendente recente (últimos 5 minutos)
        const recentNotification = await findOne('whatsapp_notifications', {
            user_id: userId,
            sent: 0
        });
        
        if (recentNotification) {
            const timeDiff = Date.now() - new Date(recentNotification.created_at).getTime();
            if (timeDiff < 5 * 60 * 1000) { // 5 minutos
                console.log('⚠️ Notificação recente já existe, pulando...');
                return false;
            }
        }
        
        // Adicionar à fila
        await insert('whatsapp_notifications', {
            user_id: userId,
            message_id: messageId,
            phone_number: phoneNumber,
            sent: 0,
            attempts: 0
        });
        
        console.log('📋 Notificação adicionada à fila:', { userId, messageId, phoneNumber });
        return true;
        
    } catch (error) {
        console.error('Erro ao adicionar notificação à fila:', error);
        return false;
    }
}

// Processar fila a cada 30 segundos
setInterval(processNotificationQueue, 30000);

// Função para obter status do WhatsApp
function getWhatsAppStatus() {
    return {
        isConnected,
        currentQR,
        session: whatsappConfig.session,
        hasClient: !!client
    };
}

// Função para desconectar WhatsApp
async function disconnectWhatsApp() {
    try {
        if (client) {
            await client.close();
            client = null;
        }
        isConnected = false;
        currentQR = null;
        await updateConnectionStatus(false);
        console.log('📱 WhatsApp desconectado');
    } catch (error) {
        console.error('Erro ao desconectar WhatsApp:', error);
    }
}

// Função para reiniciar WhatsApp
async function restartWhatsApp() {
    try {
        await disconnectWhatsApp();
        await new Promise(resolve => setTimeout(resolve, 5000)); // Aguardar 5 segundos
        await initializeWhatsApp();
    } catch (error) {
        console.error('Erro ao reiniciar WhatsApp:', error);
        throw error;
    }
}

module.exports = {
    initializeWhatsApp,
    sendWhatsAppNotification,
    checkWhatsAppNumber,
    queueNotification,
    getWhatsAppStatus,
    disconnectWhatsApp,
    restartWhatsApp,
    processNotificationQueue
};